#!/usr/bin/env python3
import argparse, json, os, sys, pathlib
from typing import Dict, Any, List

# We import plotting libs lazily to avoid import cost for --summary
def load_json(path: str) -> Dict[str, Any]:
    with open(path, "r", encoding="utf-8") as f:
        return json.load(f)

def print_summary(audit_path: str):
    from rich.console import Console
    from rich.table import Table
    c = Console()
    if not os.path.exists(audit_path):
        c.print(f"[red]audit.json not found at {audit_path}[/red]")
        sys.exit(2)
    audit = load_json(audit_path)
    # Try a few common keys; fall back to pass if found
    verdict = audit.get("verdict") or audit.get("status") or audit.get("result") or audit.get("pass")
    passed = None
    if isinstance(verdict, str):
        passed = "PASS" in verdict.upper()
    elif isinstance(verdict, bool):
        passed = verdict
    # Render
    table = Table(title="Acceptance Summary")
    table.add_column("Key")
    table.add_column("Value")
    table.add_row("File", str(audit_path))
    table.add_row("Verdict", ("PASS" if passed else "FAIL or Unknown"))
    # Dump a few top-level fields if present
    for k in ("sr", "quantum", "sensing", "audits"):
        if k in audit:
            table.add_row(k, "[dim]present[/dim]")
    c.print(table)
    # Also emit a one-liner status for CI-like logs
    if passed is True:
        c.print("[bold green]PASS[/bold green]")
        sys.exit(0)
    else:
        c.print("[bold red]FAIL or Unknown[/bold red]")
        sys.exit(1)

def try_plot_csvs(artifacts_dir: str, outdir: str):
    import pandas as pd
    import matplotlib.pyplot as plt
    os.makedirs(outdir, exist_ok=True)
    csvs = [p for p in os.listdir(artifacts_dir) if p.lower().endswith(".csv")]
    made = []
    for csv in csvs:
        path = os.path.join(artifacts_dir, csv)
        try:
            df = pd.read_csv(path)
        except Exception as e:
            continue
        # Plot each numeric column as a line over index (one figure per column)
        for col in df.columns:
            if pd.api.types.is_numeric_dtype(df[col]):
                fig = plt.figure()
                plt.plot(df.index.values, df[col].values)
                plt.title(f"{csv} — {col}")
                plt.xlabel("row")
                plt.ylabel(col)
                out = os.path.join(outdir, f"{pathlib.Path(csv).stem}__{col}.png")
                fig.savefig(out, bbox_inches="tight")
                plt.close(fig)
                made.append(out)
    return made

def try_plot_readouts(readouts_path: str, outdir: str):
    import matplotlib.pyplot as plt
    os.makedirs(outdir, exist_ok=True)
    if not os.path.exists(readouts_path):
        return []
    data = load_json(readouts_path)
    made = []
    # A very generic plotter: if we find arrays of equal length under numeric keys, plot them pairwise
    # E.g., 'alpha' vs 'gamma', or 't' vs 'x'. We try a few common pairs.
    candidates = [
        ("alpha","gamma"),
        ("t","x"),
        ("time","reach"),
        ("step","value"),
    ]
    for xk, yk in candidates:
        x = data.get(xk); y = data.get(yk)
        if isinstance(x, list) and isinstance(y, list) and len(x)==len(y) and len(x)>0:
            fig = plt.figure()
            plt.plot(x, y)
            plt.title(f"{xk} vs {yk}")
            plt.xlabel(xk); plt.ylabel(yk)
            out = os.path.join(outdir, f"{xk}_vs_{yk}.png")
            fig.savefig(out, bbox_inches="tight")
            plt.close(fig)
            made.append(out)
    return made

def main():
    parser = argparse.ArgumentParser(description="Replay helper for acceptance artifacts")
    parser.add_argument("--summary", action="store_true", help="Print a PASS/FAIL one-liner from artifacts/audit.json")
    parser.add_argument("--plots", action="store_true", help="Generate simple plots from artifacts/*.csv and readouts.json")
    parser.add_argument("--artifacts", default="artifacts", help="Artifacts directory (default: artifacts)")
    parser.add_argument("--outdir", default="artifacts/plots", help="Where to write plots (default: artifacts/plots)")
    args = parser.parse_args()

    artifacts_dir = args.artifacts
    audit_path = os.path.join(artifacts_dir, "audit.json")
    readouts_path = os.path.join(artifacts_dir, "readouts.json")

    if args.summary:
        print_summary(audit_path)

    if args.plots:
        made1 = try_plot_csvs(artifacts_dir, args.outdir)
        made2 = try_plot_readouts(readouts_path, args.outdir)
        total = len(made1) + len(made2)
        print(f"Generated {total} plot(s) under {args.outdir}")
        if total == 0:
            print("No plottable data detected; ensure CSVs contain numeric columns or readouts.json has known arrays.")

    if not args.summary and not args.plots:
        print("Nothing to do. Use --summary or --plots. See --help.")

if __name__ == '__main__':
    main()
